<?php
// /PayPal/webhook.php
require_once 'config.php';
use PayPalHttp\HttpRequest;

$payload = @file_get_contents('php://input');
$event = json_decode($payload, true);

if (!$payload || !is_array($event)) {
    http_response_code(400);
    exit('Invalid payload');
}

if (PAYPAL_WEBHOOK_ID === '') {
    error_log('Webhook Error: PAYPAL_WEBHOOK_ID is not configured.');
    http_response_code(503);
    exit('Webhook is not configured');
}

$transmissionId = $_SERVER['HTTP_PAYPAL_TRANSMISSION_ID'] ?? '';
$transmissionTime = $_SERVER['HTTP_PAYPAL_TRANSMISSION_TIME'] ?? '';
$transmissionSig = $_SERVER['HTTP_PAYPAL_TRANSMISSION_SIG'] ?? '';
$authAlgo = $_SERVER['HTTP_PAYPAL_AUTH_ALGO'] ?? '';
$certUrl = $_SERVER['HTTP_PAYPAL_CERT_URL'] ?? '';

if (
    $transmissionId === '' ||
    $transmissionTime === '' ||
    $transmissionSig === '' ||
    $authAlgo === '' ||
    $certUrl === ''
) {
    http_response_code(400);
    exit('Missing PayPal signature headers');
}

try {
    $verifyRequest = new HttpRequest('/v1/notifications/verify-webhook-signature', 'POST');
    $verifyRequest->headers['Content-Type'] = 'application/json';
    $verifyRequest->body = [
        'auth_algo' => $authAlgo,
        'cert_url' => $certUrl,
        'transmission_id' => $transmissionId,
        'transmission_sig' => $transmissionSig,
        'transmission_time' => $transmissionTime,
        'webhook_id' => PAYPAL_WEBHOOK_ID,
        'webhook_event' => $event,
    ];

    $verification = getPayPalClient()->execute($verifyRequest);
    $verificationStatus = strtoupper((string)($verification->result->verification_status ?? ''));
    if ($verificationStatus !== 'SUCCESS') {
        http_response_code(400);
        exit('Invalid webhook signature');
    }
} catch (Exception $e) {
    error_log('Webhook verification error: ' . $e->getMessage());
    http_response_code(500);
    exit('Webhook verification failed');
}

// The user already CAPTURED the payment by clicking "Pay Now" on the PayPal page
// So we just need to listen for the confirmation event.
if (($event['event_type'] ?? '') === 'CHECKOUT.ORDER.CAPTURED') {
    $resource = $event['resource'] ?? [];

    $customId = $resource['purchase_units'][0]['custom_id'] ?? '';
    $parts = explode('|', $customId, 2);
    $sourceCode = trim((string)($parts[0] ?? ''));
    $systemUrl = getWhmcsUrlBySourceCode($sourceCode);
    if ($systemUrl === '') {
        // Backward compatibility for older orders that used base64(systemUrl)|invoiceId.
        $decodedSystemUrl = base64_decode($sourceCode, true);
        $legacyUrl = normalizeWhmcsUrl($decodedSystemUrl !== false ? $decodedSystemUrl : '');
        if (isAuthorizedWhmcsUrl($legacyUrl)) {
            $systemUrl = $legacyUrl;
        }
    }
    $invoiceId = filter_var($parts[1] ?? null, FILTER_VALIDATE_INT, ['options' => ['min_range' => 1]]);

    $capture = $resource['purchase_units'][0]['payments']['captures'][0] ?? [];
    $transactionId = trim((string)($capture['id'] ?? ''));
    $paymentAmount = is_numeric($capture['amount']['value'] ?? null) ? (float)$capture['amount']['value'] : 0.0;

    if ($invoiceId && $transactionId !== '' && $paymentAmount > 0 && isAuthorizedWhmcsUrl($systemUrl)) {
        $callbackUrl = rtrim($systemUrl, '/') . '/modules/gateways/callback/rueezpaypal.php';
        $callbackData = [
            'invoiceid' => $invoiceId,
            'transid' => $transactionId,
            'amount' => number_format($paymentAmount, 2, '.', ''),
            'status' => 'success',
        ];

        if (CALLBACK_SHARED_SECRET !== '') {
            $signaturePayload = $invoiceId . '|' . $transactionId . '|' . number_format($paymentAmount, 2, '.', '');
            $callbackData['signature'] = hash_hmac('sha256', $signaturePayload, CALLBACK_SHARED_SECRET);
        }

        $ch = curl_init($callbackUrl);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($callbackData));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        $callbackResult = curl_exec($ch);
        if ($callbackResult === false) {
            error_log('Webhook callback error: ' . curl_error($ch));
        }
        curl_close($ch);
    }
}
http_response_code(200);
